// Sketch to display images on a 160 x 128 TFT

// Renders images stored in an array in program (FLASH)
// The JPEG images are stored in header files (see jpeg1.h etc)

// As well as the TFT_eSPI library:
// https://github.com/Bodmer/TFT_eSPI
// the sketch needs the JPEG Decoder library. This can be loaded via the Library Manager.
// or can be downloaded here:
// https://github.com/Bodmer/JPEGDecoder

//----------------------------------------------------------------------------------------------------

#include <SPI.h>
#include <TFT_eSPI.h>

TFT_eSPI tft = TFT_eSPI();


// JPEG decoder library
#include <JPEGDecoder.h>

// Return the minimum of two values a and b
#define minimum(a,b)     (((a) < (b)) ? (a) : (b))

// Include the sketch header file that contains the image stored as an array of bytes
// More than one image array could be stored in each header file.
#include "jpeg1.h"
#include "jpeg1_OLD.h"
#include "jpeg3.h"
#include "jpeg4.h"

// Count how many times the image is drawn for test purposes
uint32_t icount = 0;
//----------------------------------------------------------------------------------------------------

#define COUNTOF(x) (sizeof(x) / sizeof((x)[0]))
// Seed RNG once
void seedRandomOnce() {
  static bool seeded = false;
  if (!seeded) {
    #if defined(ESP32)
      randomSeed((uint32_t)esp_random());
    #else
      randomSeed(micros());
    #endif
    seeded = true;
  }
}

void printRandomMessage(const char* msgs[], size_t count,
                        int16_t x = 0, int16_t y = 0,
                        uint8_t font = 2, uint16_t color = TFT_WHITE,
                        uint8_t maxCharsPerLine = 30) {
  if (count == 0) return;

  size_t idx = random(count);
  const char* src = msgs[idx];

  tft.setTextDatum(TL_DATUM);   // top-left datum
  tft.setCursor(x, y, font);
  tft.setTextColor(color);
  tft.setTextSize(1);

  String line = "";
  String word = "";

  auto flushLine = [&]() {
    if (line.length()) {
      tft.println(line);
      line = "";
    }
  };

  for (int i = 0; ; i++) {
    char c = src[i];

    bool atEnd = (c == '\0');
    if (c == ' ' || c == '\n' || atEnd) {
      // Try placing the word into current line
      if (word.length()) {
        int needed = (line.length() ? 1 : 0) + word.length(); // +1 for space
        if ((int)line.length() + needed <= maxCharsPerLine) {
          // fits in current line
          if (line.length()) line += ' ';
          line += word;
        } else {
          // doesn't fit -> print current line, start new with whole word
          flushLine();
          // If the word itself is longer than a full line, hard-wrap it
          if ((int)word.length() > maxCharsPerLine) {
            int start = 0;
            while (start < (int)word.length()) {
              int chunkLen = min((int)maxCharsPerLine, (int)word.length() - start);
              tft.println(word.substring(start, start + chunkLen));
              start += chunkLen;
            }
          } else {
            line = word;
          }
        }
        word = "";
      }

      // Handle explicit newline
      if (c == '\n') {
        flushLine();
      }

      if (atEnd) break;  // finished processing entire string
    } else {
      // keep building word
      word += c;
    }
  }

  // Print whatever is left
  if (line.length()) tft.println(line);
}

static const char* messageArray[] = {
  "Your voice is my favorite sound.",
  "So far, every moment we've spent together has been awesome. But I promise you, that the best is yet to come.",
  "If only you knew how much those little moments with you matter to me.",
  "You have no idea how much my heart races when I see you.",
  "I love when I catch you looking at me.",
  "Thank you for being the best thing that is ever happened to me.",
  "Your smile is my favorite thing in the world.",
  "Somehow just hearing your name gives me butterflies.",
  "Since the time I've met you, I cry a little less, laugh a little harder, and smile all the more, just because I have you, my life is a better place.",
  "Every day with you is a wonderful addition to my life's journey.",
  "You are my happily ever after.",
  "You're my paradise, and I'd happily get stranded on you for a lifetime.",
  "Just when I think that it is impossible to love you any more than I already do, you prove me wrong.",
  "You are far more than my partner. You are my soulmate in every way.",
  "Every day I continue to chose you, and every day that choice gets easier and easier.",
  "Just wanted to say I love you more than words can express.",
  "Just a reminder: I am so lucky to have you.",
  "You are the reason my heart beats a little faster."
};


void setup() {
  Serial.begin(115200);
  tft.begin();
  
  tft.setSwapBytes(true);
  tft.setRotation(3);  // portrait

  tft.fillScreen(TFT_BLACK);
  tft.pushImage(0,0,240,135,MyPic);

  delay(2000);
 
  tft.fillScreen(TFT_BLACK);
  tft.pushImage(0,0,240,135,MyPic2);

  tft.setCursor(10, 10, 2);
  tft.setTextColor(TFT_WHITE);  
  tft.setTextSize(1);
  printRandomMessage(messageArray, COUNTOF(messageArray), 10, 10, 2, TFT_WHITE,30);
}

void loop() {}

void drawArrayJpeg(const uint8_t arrayname[], uint32_t array_size, int xpos, int ypos) {

  int x = xpos;
  int y = ypos;

  JpegDec.decodeArray(arrayname, array_size);
  
 
  
  renderJPEG(x, y);
  
  Serial.println("#########################");
}

void renderJPEG(int xpos, int ypos) {

  // retrieve infomration about the image
  uint16_t *pImg;
  uint16_t mcu_w = JpegDec.MCUWidth;
  uint16_t mcu_h = JpegDec.MCUHeight;
  uint32_t max_x = JpegDec.width;
  uint32_t max_y = JpegDec.height;

  // Jpeg images are draw as a set of image block (tiles) called Minimum Coding Units (MCUs)
  // Typically these MCUs are 16x16 pixel blocks
  // Determine the width and height of the right and bottom edge image blocks
  uint32_t min_w = minimum(mcu_w, max_x % mcu_w);
  uint32_t min_h = minimum(mcu_h, max_y % mcu_h);

  // save the current image block size
  uint32_t win_w = mcu_w;
  uint32_t win_h = mcu_h;

  // record the current time so we can measure how long it takes to draw an image
  uint32_t drawTime = millis();

  // save the coordinate of the right and bottom edges to assist image cropping
  // to the screen size
  max_x += xpos;
  max_y += ypos;

  // read each MCU block until there are no more
  while (JpegDec.readSwappedBytes()) {
	  
    // save a pointer to the image block
    pImg = JpegDec.pImage ;

    // calculate where the image block should be drawn on the screen
    int mcu_x = JpegDec.MCUx * mcu_w + xpos;  // Calculate coordinates of top left corner of current MCU
    int mcu_y = JpegDec.MCUy * mcu_h + ypos;

    // check if the image block size needs to be changed for the right edge
    if (mcu_x + mcu_w <= max_x) win_w = mcu_w;
    else win_w = min_w;

    // check if the image block size needs to be changed for the bottom edge
    if (mcu_y + mcu_h <= max_y) win_h = mcu_h;
    else win_h = min_h;

    // copy pixels into a contiguous block
    if (win_w != mcu_w)
    {
      uint16_t *cImg;
      int p = 0;
      cImg = pImg + win_w;
      for (int h = 1; h < win_h; h++)
      {
        p += mcu_w;
        for (int w = 0; w < win_w; w++)
        {
          *cImg = *(pImg + w + p);
          cImg++;
        }
      }
    }

    // draw image MCU block only if it will fit on the screen
    if (( mcu_x + win_w ) <= tft.width() && ( mcu_y + win_h ) <= tft.height())
    {
      tft.pushRect(mcu_x, mcu_y, win_w, win_h, pImg);
    }
    else if ( (mcu_y + win_h) >= tft.height()) JpegDec.abort(); // Image has run off bottom of screen so abort decoding
  }

  // calculate how long it took to draw the image
  drawTime = millis() - drawTime;

  // print the results to the serial port
  Serial.print(F(  "Total render time was    : ")); Serial.print(drawTime); Serial.println(F(" ms"));
  Serial.println(F(""));
}



